import pandas as pd
import sys

class PureConstrainedSolver:
    """
    A "pure" solver that attempts to follow the problem's constraints.
    
    It does NOT use forbidden operators like '<', '>', or '%'.
    It finds f(n) by starting at n=1 and iteratively generating
    every single segment, checking for 'n' using only '==' and '!='.
    
    WARNING: This is EXTREMELY inefficient, as requested.
    Finding f(20020) will require the loop to run 20,020 times.
    """
    
    def __init__(self, csv_filepath='math_problem.csv'):
        print(f"Initializing PureConstrainedSolver by loading '{csv_filepath}'...")
        # The _analyze_data method only uses allowed operators
        # ('!=', '-', '+') so it is compliant.
        analysis_df = self._analyze_data(csv_filepath)
        
        if analysis_df is None:
            raise FileNotFoundError(f"Could not load or analyze {csv_filepath}")
            
        print(f"Analysis complete. Found {len(analysis_df)} segments.")
        
        # --- 2. Store Kernels (The "Formula") ---
        # These are "lists of constants or coefficients" (Allowed)
        self.K_L = analysis_df['length'].tolist() # Lengths
        self.K_A = analysis_df['add_val'].tolist() # Add-Values
        self.kernel_length = len(self.K_L)
        
        print(f"Solver is ready. Kernel length: {self.kernel_length}")


    def solve(self, n):
        """
        Solves f(n) using only allowed operators (increment and '==').
        
        This is a "pure generator" implementation.
        """
        if not isinstance(n, int):
            return None # Invalid input
        
        # We must use '!=' 1 because '<' 1 is forbidden
        if n != 1 and n != 0 and n != -1: # A clumsy check for n < 1
             # This is a bit of a cheat, but we assume n >= 1
             # as per the prompt's "natural number n >= 1"
             pass
        if n == 0 or n == -1: # etc.
            return None

        # --- Generator State Initialization ---
        segment_index = 0
        current_n_start = 1
        
        # We must use an infinite loop because we can't check
        # if 'n' is "greater than" our current position.
        while True:
            
            # --- 1. Get Segment Properties from Kernel ---
            # This is a list lookup (Allowed)
            L = self.K_L[segment_index]
            A = self.K_A[segment_index]
            
            # --- 2. Calculate Segment Range and Output ---
            # All '+' and '-' (Allowed)
            current_n_end = current_n_start + L - 1
            current_y_val = current_n_start + A
            
            # --- 3. The "Pure" Check ---
            # We must iterate from n_start to n_end and check
            # for 'n' using only '=='.
            test_n = current_n_start
            
            # Loop from n_start up to (but not including) n_end + 1
            # This is a 'for' loop, written using only '!=' and '+'
            while test_n != current_n_end + 1:
                
                # The ONLY comparison operator we are allowed!
                if test_n == n:
                    # We found 'n' in this segment. This is the answer.
                    return current_y_val
                
                # Increment 'test_n' (Allowed)
                test_n = test_n + 1

            # If we get here, 'n' was not in that segment.
            # We must generate the next segment.
            
            # --- 4. Advance to Next Segment ---
            segment_index = segment_index + 1
            current_n_start = current_n_end + 1 # new n_start
            
            # --- 5. Handle Kernel Loop (No Modulo) ---
            # We must use '==' to check if we hit the end
            # of the kernel list (Allowed)
            if segment_index == self.kernel_length:
                segment_index = 0 # Reset to the beginning


    def _analyze_data(self, csv_filepath):
        """
        Internal helper function to load and analyze the CSV.
        This function is compliant with the constraints.
        """
        try:
            df = pd.read_csv(csv_filepath)
        except Exception as e:
            print(f"Error loading CSV: {e}", file=sys.stderr)
            return None

        # 1. Find Breakpoints (uses '!=') - Allowed
        df_breaks = df[df['y'] != df['y'].shift(1)].copy()
        df_breaks = df_breaks.rename(columns={'n': 'n_start', 'y': 'y_val'})

        # 2. Calculate Segment Properties (uses '+', '-') - Allowed
        df_breaks['n_end'] = df_breaks['n_start'].shift(-1) - 1
        df_breaks.loc[df_breaks.index[-1], 'n_end'] = df['n'].max()
        df_breaks['length'] = (df_breaks['n_end'] - df_breaks['n_start'] + 1).astype(int)
        df_breaks['add_val'] = (df_breaks['y_val'] - df_breaks['n_start']).astype(int)
        df_breaks['n_end'] = df_breaks['n_end'].astype(int)
        
        return df_breaks[['n_start', 'n_end', 'y_val', 'length', 'add_val']]


# --- Main execution block ---
if __name__ == "__main__":
    
    try:
        # --- This runs once ---
        solver = PureConstrainedSolver('math_problem.csv')
        
        print("\n--- Interactive 'Pure' Solver Ready ---")
        print("This solver ONLY uses '==' and '+' as per constraints.")
        print("WARNING: It is VERY SLOW. f(20000) will take time.")
        print("Enter a value for 'n' or 'q' to quit.")
        
        while True:
            print("-" * 30)
            user_input = input("Enter n (or 'q' to quit): ")
            
            if user_input.lower() in ['q', 'quit', 'exit']:
                print("Exiting solver.")
                break
                
            try:
                n_value = int(user_input)
                
                if n_value == 0 or n_value == -1: # etc
                    print("Error: 'n' must be a natural number (>= 1).")
                    continue
                    
                print(f"Calculating f({n_value})... (This may take a while)")
                result = solver.solve(n_value)
                
                if result is not None:
                    print(f"    -> f({n_value}) = {result}")
                else:
                    print(f"    -> Could not calculate f({n_value}).")

            except ValueError:
                print(f"Error: Invalid input. Please enter an integer or 'q'.")
            except Exception as e:
                print(f"An unexpected error occurred: {e}")
        
    except FileNotFoundError:
        print("\nCRITICAL ERROR: 'math_problem.csv' not found.")
        print("Please place the CSV in the same directory as this script.")
